import streamlit as st
import pandas as pd
import numpy as np
import os
import matplotlib.pyplot as plt
import tiktoken
import random
import joblib
import json
import csv
from transformers import pipeline
import keras
from tensorflow.keras.preprocessing.sequence import pad_sequences
from sklearn.preprocessing import LabelEncoder
from sklearn.feature_extraction.text import CountVectorizer
from tensorflow.keras.utils import plot_model
from filesplit.merge import Merge
from extra_streamlit_components import tab_bar, TabBarItemData

title = "Identification de langue"
sidebar_name = "Identification de langue"


# CountVectorizer a une liste de phrase en entrée.
# Cette fonction met les données d'entrée dans le bon format
def format_to_vectorize(data):
    X_tok = []
    if "DataFrame" in str(type(data)):sentences = data.tolist()
    elif "str" in str(type(data)):
        sentences =[data]
    else: sentences = data
                          
    for sentence in sentences:
        X_tok.append(sentence) 
    return X_tok

def create_BOW(data):
    global vectorizer
    
    X_tok = format_to_vectorize(data)
    X = vectorizer.transform(X_tok)
    return X

def load_vectorizer(tokenizer):
    global dict_token, dict_ids, nb_token
    
    path = 'data/vectorizer_tiktoken_big.pkl'
    vectorizer = joblib.load(path)
    dict_token = {tokenizer.decode([cle]): cle for cle, valeur in vectorizer.vocabulary_.items()}
    dict_ids = {cle: tokenizer.decode([cle]) for cle, valeur in vectorizer.vocabulary_.items()} #dict_ids.items()}
    nb_token = len(vectorizer.vocabulary_)
    return vectorizer

def lang_id_nb(sentences):
    global lan_to_language

    if "str" in str(type(sentences)):
        return lan_to_language[clf_nb.predict(create_BOW(sentences))[0]]
    else: return [lan_to_language[l] for l in clf_nb.predict(create_BOW(sentences))]

@st.cache_resource
def init_nb_identifier():
    
    tokenizer = tiktoken.get_encoding("cl100k_base")
    
    # Chargement du classificateur sauvegardé
    clf_nb = joblib.load("data/id_lang_tiktoken_nb_sparse_big.pkl")
    vectorizer = load_vectorizer(tokenizer)

    # Lisez le contenu du fichier JSON
    with open('data/multilingue/lan_to_language.json', 'r') as fichier:
        lan_to_language = json.load(fichier)
    return tokenizer, dict_token, dict_ids, nb_token, lan_to_language, clf_nb, vectorizer

def encode_text(textes):
    global tokenizer

    max_length=250
    sequences = tokenizer.encode_batch(textes)
    return pad_sequences(sequences, maxlen=max_length, padding='post')

def read_list_lan():

    with open('data/multilingue/lan_code.csv', 'r') as fichier_csv:
        reader = csv.reader(fichier_csv)
        lan_code = next(reader)
        return lan_code

@st.cache_resource
def init_dl_identifier():

    label_encoder = LabelEncoder()
    label_encoder.fit(read_list_lan())
    merge = Merge("data/dl_id_lang_split",  "./data", "dl_tiktoken_id_language_model.h5").merge(cleanup=False)
    dl_model = keras.models.load_model("data/dl_tiktoken_id_language_model.h5")
    return dl_model, label_encoder

def lang_id_dl(sentences):
    global dl_model, label_encoder
    
    if "str" in str(type(sentences)): predictions = dl_model.predict(encode_text([sentences]))
    else:  predictions = dl_model.predict(encode_text(sentences))
    # Décodage des prédictions en langues
    predicted_labels_encoded = np.argmax(predictions, axis=1)
    predicted_languages = label_encoder.classes_[predicted_labels_encoded]
    if "str" in str(type(sentences)): return lan_to_language[predicted_languages[0]]
    else: return [l for l in predicted_languages]

@st.cache_resource
def init_lang_id_external():
    lang_id_model_ext = pipeline('text-classification',model="papluca/xlm-roberta-base-language-detection")
    dict_xlmr  = {"ar":"ara", "bg":"bul", "de":"deu", "el": "ell", "en":"eng", "es":"spa", "fr":"fra", "hi": "hin","it":"ita","ja":"jpn", \
                  "nl":"nld", "pl":"pol", "pt":"por", "ru":"rus", "sw":"swh", "th":"tha", "tr":"tur", "ur": "urd", "vi":"vie", "zh":"cmn"}
    return lang_id_model_ext, dict_xlmr

def run():
    global tokenizer, vectorizer, dict_token, dict_ids, nb_token, lan_to_language, clf_nb
    global dl_model, label_encoder
   

    tokenizer, dict_token, dict_ids, nb_token, lan_to_language, clf_nb, vectorizer = init_nb_identifier()
    dl_model, label_encoder = init_dl_identifier()
    lang_id_model_ext, dict_xlmr = init_lang_id_external()

    st.write("")
    st.title(title)
    st.write("## **Explications :**\n")
    st.markdown(
        """
        Afin de mettre en oeuvre cette fonctionnalité nous avons utilisé un jeu d'entrainement multilinge de 9.757.778 phrases dans 95 langues.  
        Puis, nous avons utilisé 2 méthodes pour identifier la langue d'un texte:  
        1. un classificateur **Naïve Bayes**  
        2. un modèle de **Deep Learning**  

        Les 2 modèles ont un accuracy similaire sur le jeu de test: **:red[96% pour NB et 97,5% pour DL]**  
        """
    )

    chosen_id = tab_bar(data=[
        TabBarItemData(id="tab1", title="Id. Naïve Bayes", description="avec le Bag Of Words"),
        TabBarItemData(id="tab2", title="Id. Deep Learning", description=" avec Keras"),
        TabBarItemData(id="tab3", title="Interpretabilité", description="du modèle Naïve Bayes ")],
        default="tab1")
    
    if (chosen_id == "tab1") or (chosen_id == "tab2"):
        st.write("## **Paramètres :**\n")
        custom_sentence = st.text_area(label="Saisir le texte dont vous souhaitez identifier la langue:")
        st.button(label="Valider", type="primary")
        if custom_sentence!='':
            st.write("## **Résultats :**\n")
            st.markdown(
                """
                |Identifieur                          |Langue détectée|
                |-------------------------------------|---------------|
                |classificateur Naïve Bayes           |**:red["""+lang_id_nb(custom_sentence)+"""]**|
                |le modèle de Deep Learning           |**:red["""+lang_id_dl(custom_sentence)+"""]**|
                |XLM-RoBERTa (Hugging Face)           |**:red["""+lan_to_language[dict_xlmr[lang_id_model_ext(custom_sentence)[0]['label']]]+"""]**|
                """
                )
        st.write("## **Details sur la méthode :**\n")
        if (chosen_id == "tab1"):
            st.markdown(
                """
                Afin d'utiliser le classificateur Naïve Bayes, il nous a fallu:
                - Créer un Bag of Words de token..
                - ..Tokeniser le texte d'entrainement avec CountVectorizer et un tokenizer 'custom', **Tiktoken** d'OpenAI.
                - Utiliser des matrices creuses (Sparse Matrix), car notre BOW contenait 10 M de lignes x 59122 tokens.
                - Sauvegarder le vectorizer (non serialisable) et le classificateur entrainé.  

                L'execution de toutes ces étapes est assez rapide: une dizaine de minutes  
                <br>
                Le résultat est très bon: Accuracy sur le jeu de test de 
                **:red[96%]** sur les 95 langues, et **:red[99,1%]** sur les 5 langues d'Europe de l'Ouest (en,fr,de,it,sp)  
                <br>
                **Note:** Le modèle *XLM-RoBERTa* de Hugging Face (qui identifie 20 langues seulement) a une accuracy sur notre jeu de test = **99,6%**
                """
            , unsafe_allow_html=True)
        else:
            st.markdown(
                """
                Nous avons mis en oeuvre un modèle Keras avec une couche d'embedding et 4 couches denses *(Voir architecture ci-dessous)*.  
                Nous avons utilé le tokensier **Tiktoken** d'OpenAI.  
                La couche d'embedding accepte 250 tokens, ce qui signifie que la détection de langue s'effectue sur approximativement les 200 premiers mots.  
                <br>
                Les 2 modèles ont un accuracy similaire sur le jeu de test: **:red[96% pour NB et 97,5% pour DL]**
                """
                , unsafe_allow_html=True)
            st.write("<center><h5>Architecture du modèle utilisé:</h5></center>", unsafe_allow_html=True)
            plot_model(dl_model, show_shapes=True, show_layer_names=True, show_layer_activations=True,rankdir='TB',to_file='./assets/model_plot.png')
            col1, col2, col3 = st.columns([0.15,0.7,0.15])
            with col2:
                 st.image('./assets/model_plot.png',use_column_width="auto")
    elif (chosen_id == "tab3"):
        st.write("## **Interpretabilité**\n")
        st.write("****Waiting for code from Olivier****")

    
